/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.telegram.messenger.exoplayer2.upstream;

import android.net.Uri;
import android.util.Base64;
import org.telegram.messenger.exoplayer2.C;
import org.telegram.messenger.exoplayer2.ParserException;
import java.io.IOException;
import java.net.URLDecoder;

/**
 * A {@link DataSource} for reading data URLs, as defined by RFC 2397.
 */
public final class DataSchemeDataSource implements DataSource {

  public static final String SCHEME_DATA = "data";

  private DataSpec dataSpec;
  private int bytesRead;
  private byte[] data;

  @Override
  public long open(DataSpec dataSpec) throws IOException {
    this.dataSpec = dataSpec;
    Uri uri = dataSpec.uri;
    String scheme = uri.getScheme();
    if (!SCHEME_DATA.equals(scheme)) {
      throw new ParserException("Unsupported scheme: " + scheme);
    }
    String[] uriParts = uri.getSchemeSpecificPart().split(",");
    if (uriParts.length > 2) {
      throw new ParserException("Unexpected URI format: " + uri);
    }
    String dataString = uriParts[1];
    if (uriParts[0].contains(";base64")) {
      try {
        data = Base64.decode(dataString, 0);
      } catch (IllegalArgumentException e) {
        throw new ParserException("Error while parsing Base64 encoded string: " + dataString, e);
      }
    } else {
      // TODO: Add support for other charsets.
      data = URLDecoder.decode(dataString, C.ASCII_NAME).getBytes();
    }
    return data.length;
  }

  @Override
  public int read(byte[] buffer, int offset, int readLength) {
    if (readLength == 0) {
      return 0;
    }
    int remainingBytes = data.length - bytesRead;
    if (remainingBytes == 0) {
      return C.RESULT_END_OF_INPUT;
    }
    readLength = Math.min(readLength, remainingBytes);
    System.arraycopy(data, bytesRead, buffer, offset, readLength);
    bytesRead += readLength;
    return readLength;
  }

  @Override
  public Uri getUri() {
    return dataSpec != null ? dataSpec.uri : null;
  }

  @Override
  public void close() throws IOException {
    dataSpec = null;
    data = null;
  }

}
