const fs = require('fs');
const path = require('path');
const AWS = require('aws-sdk');
const { Client } = require('ssh2');
const axios = require('axios');
const downloadsPath = path.join(process.env.HOME || process.env.USERPROFILE, 'Downloads');
const rootKeyFilePath = path.join(downloadsPath, 'rootkey.csv');

// Check if the rootkey.csv file exists
if (!fs.existsSync(rootKeyFilePath)) {
    console.error(`File not found: ${rootKeyFilePath}`);
    process.exit(1);
}

// Function to read the second line and extract AWS credentials
function extractAWSCredentials(filePath) {
    const fileContent = fs.readFileSync(filePath, 'utf8');
    const lines = fileContent.split('\n');

    if (lines.length < 2) {
        throw new Error('The rootkey.csv file does not contain enough lines.');
    }

    const secondLine = lines[1].trim();
    const [accessKeyId, secretAccessKey] = secondLine.split(',');

    if (accessKeyId && secretAccessKey) {
        return {
            accessKeyId: accessKeyId.trim(),
            secretAccessKey: secretAccessKey.trim(),
        };
    } else {
        throw new Error('Failed to extract AWS credentials from the second line of rootkey.csv');
    }
}

// Function to check if instance exists
async function checkInstanceExists(lightsail, instanceName) {
    try {
        const instances = await lightsail.getInstances().promise();
        return instances.instances.some(instance => instance.name === instanceName);
    } catch (error) {
        console.error(`Error fetching instances: ${error.message}`);
        return false;
    }
}

// Function to create a single Lightsail instance
async function createLightsailInstance(lightsail, region, instanceName) {
    const instanceParams = {
        blueprintId: 'ubuntu_20_04', // Ubuntu 20.04
        bundleId: 'large_2_0', // $44 USD plan for Lightsail
        availabilityZone: `${region}a`, // Availability zone (usually 'a', 'b', or 'c')
        instanceNames: [instanceName], // Only creating one instance at a time
    };

    try {
        const result = await lightsail.createInstances(instanceParams).promise();
        console.log(`Created instance ${instanceName} in region ${region}:`, result);
    } catch (error) {
        console.error(`Failed to create instance ${instanceName} in ${region}:`, error.message);
    }
}

// Function to download Lightsail default SSH key for each region
async function downloadSSHKey(region) {
    const keyFilePath = path.join(downloadsPath, `LightsailDefaultKey-${region}.pem`);


    const lightsail = new AWS.Lightsail({ region });
    try {
        const keyData = await lightsail.getKeyPair({ keyPairName: `LightsailDefaultKey-${region}` }).promise();
        fs.writeFileSync(keyFilePath, keyData.keyMaterial, { mode: 0o600 }); // Write the key with proper permissions
        console.log(`Downloaded SSH key for region ${region}`);
        return keyFilePath;
    } catch (error) {
        console.error(`Failed to download SSH key for region ${region}:`, error.message);
        return null;
    }
}

// Function to SSH into instance to validate SSH key

// Function to add delay
const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

async function main() {

        // Extract credentials
        const credentials = extractAWSCredentials(rootKeyFilePath);
        console.log('AWS Credentials Loaded:', credentials);

        // Set AWS config globally
        AWS.config.update({
            accessKeyId: credentials.accessKeyId,
            secretAccessKey: credentials.secretAccessKey,
        });

        // Define the regions and required instances
        let regions = ['us-west-2', 'us-east-1', 'us-east-2'];
        let requiredInstances = ['Ubuntu-1', 'Ubuntu-2'];

        // Loop through each region and check/create instances
        for (const region of regions) {
            const lightsail = new AWS.Lightsail({ region });

            for (const instanceName of requiredInstances) {
                const exists = await checkInstanceExists(lightsail, instanceName);
                if (!exists) {
                    console.log(`Instance ${instanceName} does not exist in region ${region}, creating...`);
                    await createLightsailInstance(lightsail, region, instanceName);
                    console.log(`Waiting 60 seconds before creating the next instance...`);
                    await sleep(60000); // Wait for 60 seconds to avoid hitting rate limits
                } else {
                    console.log(`Instance ${instanceName} already exists in region ${region}, skipping creation.`);
                }
            }
        }

        // Download and verify SSH keys for all regions after creating instances


        console.log('All Done');


}

main().catch(console.error);
